/*****************************************************************************/
/*  Includes                                                                 */
/*****************************************************************************/
#include "netx51_uart.h" /* function prototypes and defines */
#include "hal_resources_defines_netx51.h"
#include <stdarg.h>      /* uprintf */

/*****************************************************************************/
/*  Variables                                                                */
/*****************************************************************************/
__USE_UART

/*****************************************************************************/
/* Definitions                                                               */
/*****************************************************************************/
/* DEV_BAUDRATE is 100 times to small -> multiply with 100 (or divide by DEV_FREQUENCY/100) */
#define NX51_DEV_BAUDRATE_DIV_LO(baud) (((baud*16*65536)/(NX51_DEV_FREQUENCY/100)) & 0xff)
#define NX51_DEV_BAUDRATE_DIV_HI(baud) (((baud*16*65536)/(NX51_DEV_FREQUENCY/100))>>8)

/*****************************************************************************/
/*! Initialize serial Port
* \description
*   Initialize serial port.
* \class 
*   UART 
* \params
*   uUartNr               [in]  Number of UART
*   eBaud                 [in]  Baud rate
* \return
*                                                                            */
/*****************************************************************************/
void NX51_UART_Init( unsigned int         uUartNr,
                     NX51_UART_BAUDRATE_E eBaud )
{
  NX51_UART_AREA_T* ptUart = s_aptUart[uUartNr];
    
  /* First of all disable everything */
  ptUart->ulUartcr     = 0;
  
  /* Set the bit for the second baud rate mode */
  ptUart->ulUartcr_2   = MSK_NX51_uartcr_2_Baud_Rate_Mode;
  
  /* Adjust the baud rate register */
  ptUart->ulUartlcr_l  = NX51_DEV_BAUDRATE_DIV_LO((uint32_t)eBaud);
  ptUart->ulUartlcr_m  = NX51_DEV_BAUDRATE_DIV_HI((uint32_t)eBaud);
  
  /* set UART to 8 bits, 1 stop bit, no parity, FIFO enabled */
  ptUart->ulUartlcr_h  = (MSK_NX51_uartlcr_h_WLEN | MSK_NX51_uartlcr_h_FEN);
  /* Set TX-Driver to enabled */
  ptUart->ulUartdrvout = MSK_NX51_uartdrvout_DRVTX;
  /* Finally enable the UART */
  ptUart->ulUartcr     = MSK_NX51_uartcr_uartEN;
}

/*****************************************************************************/
/*! Close Serial Port
* \description
*   Closes serial port.
* \class 
*   UART 
* \params
*   uUartNr              [in]  Number of UART
* \return
*                                                                            */
/*****************************************************************************/
void NX51_UART_CloseSerial( unsigned int uUartNr )
{
  NX51_UART_AREA_T* ptUart = s_aptUart[uUartNr];
  volatile uint32_t ulFoo;
  unsigned int uCnt;
   
  while((ptUart->ulUartfr & MSK_NX51_uartfr_BUSY)!=0); /* UART busy ? */
  ptUart->ulUartcr     = 0; /* First of all disable everything */
  ptUart->ulUartlcr_m  = 0; /* Adjust the baud rate register */
  ptUart->ulUartlcr_l  = 0;
  ptUart->ulUartlcr_h  = 0; /* UART to be 8 bits, 1 stop bit, no parity, FIFO enabled */
  ptUart->ulUartdrvout = 0; /* Set TX-Driver to disabled */
  for(uCnt=0; uCnt < 5; uCnt++)
  {
    ulFoo = ptUart->ulUartdr; /* Read out the reception register if there are still bytes in it */
  }
}

/*****************************************************************************/
/*! Put Character
* \description
*   Writes a character into UART FIFO.
* \class 
*   UART 
* \params
*   uUartNr               [in]  Number of UART
*   bChar                 [in]  Character Content
* \return
*                                                                            */
/*****************************************************************************/
void NX51_UART_PutCharacter( unsigned int  uUartNr,
                             const uint8_t bChar )
{
  NX51_UART_AREA_T* ptUart = s_aptUart[uUartNr];
          
  /* Wait until there is space in the FIFO */
  while( (ptUart->ulUartfr & MSK_NX51_uartfr_TXFF) !=0 ) {};
  ptUart->ulUartdr = bChar;
}

/*****************************************************************************/
/*! Get Character
* \description
*   Read a character from UART FIFO.
* \class 
*   UART 
* \params
*   uUartNr              [in]  Number of UART
* \return
*   Received character
*   -1 on erroneous                                                          */
/*****************************************************************************/
int NX51_UART_GetCharacter( unsigned int uUartNr )
{
  uint32_t ulDat;
  NX51_UART_AREA_T* ptUart = s_aptUart[uUartNr];
                 
  /* We need to wait a defined time for a character */
  while( (ptUart->ulUartfr & MSK_NX51_uartfr_RXFE) != 0 ); /* Wait infinite for new data in the FIFO */
                 
  ulDat = ptUart->ulUartdr;  /* Get the received byte */
  if( (ulDat & 0xFFFFFF00) != 0 ) { /* we had a receive error */
    ptUart->ulUartrsr = 0; /* Clear the error */
    return -1; /* return error */
  }  else {
    return (int)ulDat;
  }       
}

/*****************************************************************************/
/*! UART Peek
* \description
*   -
* \class 
*   UART 
* \params
*   uUartNr              [in]  Number of UART
* \return
*   0
*   1                                                                        */
/*****************************************************************************/
int NX51_UART_Peek( unsigned int uUartNr )
{
  NX51_UART_AREA_T* ptUart = s_aptUart[uUartNr];
  
  return ( (ptUart->ulUartfr & MSK_NX51_uartfr_RXFE) == 0 );
}

/*****************************************************************************/
/*! UART Printf
* \description
*   This version of printf is very limited compared to the ANSI version.
*   It can only handle the following conversions:
*    {length}   :  Interpret the argument as 'unsigned integer' and print it in hexadecimal.
*                   The 'length' parameter is optional. It must be one single decimal digit
*                   starting with '1' (i.e. '0' is no valid length)
*   Notes:
*    Passing a NULL pointer as the format string results in the output "NULL\n"
*    The routine does not know escaping the format identifier. This means it is not possible
*    to print '' characters.
*    The default length for hexadecimal output is 8 digits.
* \class 
*   UART 
* \params
*   pchFmt              [in]   Pointer of Character to put out
* \return
*                                                                            */
/*****************************************************************************/
void NX51_UART_UPrintf( const char* pchFmt, ... )
{
  int iLen;
  int iLenParam;
  unsigned int uDigit;
  char cChar;
  unsigned int ulArgument;
  va_list ptArgument;
  
  /* Get initial pointer to first argument */
  va_start(ptArgument, pchFmt);
  
  /* Is it a NULL Pointer ? */
  if( pchFmt==(char*)0 ) {
    pchFmt = "NULL\n";
  }
  
  while( (cChar=*(pchFmt++))!=0 ) {
    /* is there a format command? */
    if( cChar!='') {
      /* no format command -> direct output */
      NX51_UART_PutCharacter(0, cChar);
    } else {
      /* get the argument */
      ulArgument = va_arg((ptArgument),unsigned int);
    
      /* default length is 8 digits */
      iLen = 28;
    
      /* get the optional length */
      cChar = *pchFmt;
      iLenParam = cChar-'1';
      if( iLenParam>=0 && iLenParam<9 ) {
        iLen = iLenParam<<2;
        ++pchFmt;
      }
    
      do {
        uDigit = ulArgument>>iLen;
        uDigit &= 0x0f;
        if( uDigit>9 ) {
          uDigit += 'A'-('9'+1);
        }
        uDigit += '0';
        NX51_UART_PutCharacter(0, uDigit);
        iLen -= 4;
      } while( iLen>=0 );
    }
  }
}

/*****************************************************************************/
/*! UART Print a Decimal Value
* \description
*   Output a decimal value on UART port.
* \class 
*   UART 
* \params
*   pchFmt              [in]  Pointer of Character to put out
* \return
*                                                                            */
/*****************************************************************************/
void NX51_UART_UPrintfDec( const char* pchFmt, ... )
{
  int iDivider;
  unsigned int uDigit;
  char cChar;
  unsigned int ulArgument;
  va_list ptArgument;

  /* Get initial pointer to first argument */
  va_start(ptArgument, pchFmt);

  /* Is it a NULL Pointer ? */
  if( pchFmt==(char*)0 ) {
    pchFmt = "NULL\n";
  }

  while( (cChar=*(pchFmt++))!=0 ) {
    /* is there a format command? */
    if( cChar!='') {
      /* no format command -> direct output */
      NX51_UART_PutCharacter(0, cChar);
    } else {
      /* get the argument */
      ulArgument = va_arg((ptArgument),unsigned int);

      /* determine start value */
      iDivider = 1000000000;
      if( ulArgument != 0 ) {
        while ((ulArgument / iDivider) == 0)
        {
           iDivider = iDivider / 10;
        }
      } else {
        iDivider = 0;
      }

      /* print digits */
      do {
        uDigit = (ulArgument / iDivider);
        uDigit += '0';
         NX51_UART_PutCharacter(0, uDigit);
         ulArgument = ulArgument % iDivider;
        iDivider = iDivider / 10;
      } while( iDivider>0 );
    }
  }
}
